//
// This file is part of MorphoDynamX - http://www.MorphoDynamX.org
// Copyright (C) 2012-2015 Richard S. Smith and collaborators.
//
// If you use MorphoDynamX in your work, please cite:
//   http://dx.doi.org/10.7554/eLife.05864
//
// MorphoDynamX is free software, and is licensed under under the terms of the 
// GNU General (GPL) Public License version 2.0, http://www.gnu.org/licenses.
//
#ifndef XPIWITProcess_HPP
#define XPIWITProcess_HPP

#include <XPIWITConfig.hpp>
#include <Process.hpp>
#include <Information.hpp>
#include <Progress.hpp>

namespace mgx
{
  ///\addtogroup Process
  ///@{
  /**
   * \class TWANGSegment XPIWITProcess.hpp <XPIWITProcess.hpp>
   *
   * Use XPIWIT with to do a TWANG nuclei segmentation 
   */
  class XPIWIT_EXPORT TwangSegment : public Process 
  {
  public:
    TwangSegment(const Process &process) : Process(process) 
    {
      setName("Stack/XPIWIT/TWANG Nuclei Segment");
      setDesc("Use XPIWIT to perform Nuclei segmentation with the TWANG pipeline.\n"
              "If you use this process in your work, please cite:\n"
              "  - Bartschat et al. 2016 XPIWIT - An XML Pipeline Wrapper for the Insight Toolkit, Bioinformatics 32(2).\n"
              "  - Stegmaier et al. 2014 Fast Segmentation of Stained Nuclei in Terabyte-Scale,\n"
              "    Time Resolved 3D Microscopy Image Stacks, PLoS One. 2014 9(2) e90036.");
      setIcon(QIcon(":/images/XPIWIT.png"));
      
      addParm("Min Sigma", "Minimum sigma for LoGScaleSpaceMaximumProjectionFilter", "4.0");
      addParm("Max Sigma", "Maximum sigma for LoGScaleSpaceMaximumProjectionFilter", "5.0");
      addParm("Std Dev Mult", "Multiplier for the weighting kernel standard deviation for ExtractLocalExtremaFilter", "1.0");
      addParm("Neighborhood Radius", "The neighborhood radius of the local extrema search for ExtractLocalExtremaFilter", "3");
	  }
  
    bool run()
    {
      Stack* stack = currentStack();
      Store* input = currentStack()->currentStore();
      Store* output = currentStack()->work();
      return run(stack, input, output);
    }
  
    bool run(Stack *stack, Store* input, Store* output);
  };

  /**
   * \class XPIWITRun XPIWITProcess.hpp <XPIWITProcess.hpp>
   *
   * Run XPIWIT 
   */
  class XPIWIT_EXPORT XPIWITRun : public Process 
  {
  public:
    XPIWITRun(const Process &process) : Process(process) 
    {
      setName("Stack/XPIWIT/XPIWIT Run");
      setDesc("Run XPIWIT in graphical mode to create your own ITK pipelines.\n"
              "If you use XPIWIT in your work, please cite:\n"
              "  Bartschat et al. 2016 XPIWIT - An XML Pipeline Wrapper for the Insight Toolkit, Bioinformatics 32(2).");
      setIcon(QIcon(":/images/XPIWIT.png"));
	  }
  
    bool run();
  };
  
  /**
   * \class XPIWITHelp XPIWITProcess.hpp <XPIWITProcess.hpp>
   *
   * Display XPIWIT help file 
   */
  class XPIWIT_EXPORT XPIWITHelp : public Process 
  {
  public:
    XPIWITHelp(const Process &process) : Process(process) 
    {
      setName("Stack/XPIWIT/XPIWIT Help");
      setDesc("View XPIWIT user guide.\n"
              "If you use XPIWIT in your work, please cite:\n"
              "  Bartschat et al. 2016 XPIWIT - An XML Pipeline Wrapper for the Insight Toolkit, Bioinformatics 32(2).");
      setIcon(QIcon(":/images/XPIWIT.png"));
	  }
  
    bool run();
  };

  ///@}
}
#endif
