//
// This file is part of MorphoGraphX - http://www.MorphoGraphX.org
// Copyright (C) 2012-2015 Richard S. Smith and collaborators.
//
// If you use MorphoGraphX in your work, please cite:
//   http://dx.doi.org/10.7554/eLife.05864
//
// MorphoGraphX is free software, and is licensed under under the terms of the 
// GNU General (GPL) Public License version 2.0, http://www.gnu.org/licenses.
//

// Draw the cutting surface. 

#ifndef CUT_SURF_H
#define CUT_SURF_H

#include <Config.hpp>
#include <GL.hpp>

#include <Color.hpp>
#include <CuttingSurface.hpp>
#include <Geometry.hpp>
#include <MGXViewer/qglviewer.h>
#include <Parms.hpp>

namespace mgx 
{
  typedef Color<float> Color3f;
  class Shader;
  class ImgData;
  class Stack;
  
  // Class to handle rotatable cutting plane.
  class mgx_EXPORT CutSurf : public QObject 
  {
    Q_OBJECT
  public:
    CutSurf();
    ~CutSurf();
  
    CuttingSurface* cut;
  
    std::set<uint> selectV;   // List of selected vertices
  
  protected:
    uint BlendSlices;
    uint Material;
    float LineWidth;
  
    void drawSurface(ImgData& stk, bool select);
  
  public:
    // Read clipping plane parameters
    void readParms(Parms& parms, QString section);
  
    // write parms to file
    void writeParms(QTextStream& out, QString section);
  
    // Draw cutting plane
    void drawCutSurf(ImgData& stk, bool select, Shader* shader = 0);
    void drawCutSurfGrid(ImgData& stk);
  
    // Get frame
    qglviewer::ManipulatedFrame* getFrame() { return &cut->frame(); }
  
    // Clear selection
    void clearSelect();
  
    // Return true if the cutting surface is opaque if drawn for this stack
    bool showOpaqueSurface(const ImgData& stk);
  
    // Return true if the cutting surface is transparent if drawn for this stack
    bool showTransparentSurface(const ImgData& stk);
  
    float getSceneRadius() {
      return SceneRadius;
    };
  
  protected:
    double SceneRadius;
    float getSize(int val);
  
  public slots:
    // Set sizes
    void DrawCutSurfSlot(bool val);
    void ThreeAxisSlot(bool val);
    void CutSurfGridSlot(bool val);
    void CutSurfPlaneSlot(bool val);
    void CutSurfBezierSlot(bool val);
    void SizeXSlot(int val);
    void SizeYSlot(int val);
    void SizeZSlot(int val);
    void Reset(double sceneRadius);
    void setSceneBoundingBox(const Point3f& bbox);
  
  signals:
    void DrawCutSurfSignal(bool val);
    void ThreeAxisSignal(bool val);
    void CutSurfGridSignal(bool val);
    void CutSurfPlaneSignal(bool val);
    void CutSurfBezierSignal(bool val);
    void SizeXSignal(int val);
    void SizeYSignal(int val);
    void SizeZSignal(int val);
    void ViewerUpdateSignal();
  };
}
#endif
