//
// This file is part of MorphoGraphX - http://www.MorphoGraphX.org
// Copyright (C) 2012-2015 Richard S. Smith and collaborators.
//
// If you use MorphoGraphX in your work, please cite:
//   http://dx.doi.org/10.7554/eLife.05864
//
// MorphoGraphX is free software, and is licensed under under the terms of the
// GNU General (GPL) Public License version 2.0, http://www.gnu.org/licenses.
//
#ifndef PROCESS_DOCS_DIALOG_HPP
#define PROCESS_DOCS_DIALOG_HPP

#include <Config.hpp>
#include <Process.hpp>

#include <QDialog>
#include <QFileDialog>

#include <memory>
#include <QMessageBox>
#include <QCheckBox>

#include <GraphUtils.hpp>
#include <MeshProcessExport.hpp>
#include <ui_ManageAttrDlg.h>

#include <ui_PlyCellGraphDlg.h>

class QTreeWidget;
class QTreeWidgetItem;

namespace mgx
{
  class mgxBase_EXPORT ManageAttributes : public Process
  {
    Q_OBJECT
    Ui_ManageAttrDlg ui;
    public:
      ManageAttributes(const Process &process) : Process(process)
      {
	    setName("Mesh/Attributes/Manage Attributes");
	    setDesc("Manage Mesh Attributes");
	    setIcon(QIcon(":/images/Manage.png"));

	    addParm("Mesh number","Mesh number","-1");
	   }

      bool initialize(QWidget *parent);
      bool run() { return true; }

      bool setUpTree(const QStringList &attr);

    protected slots:
      void on_attrTreeWidget_itemClicked(QTreeWidgetItem *, int);
      void on_clearPushButton_clicked();
      void on_clearAllPushButton_clicked();
      void on_savePushButton_clicked();

    protected:
      Attributes *attributes;
  };


  class mgxBase_EXPORT SaveAttributesCSV : public Process
  {
  Q_OBJECT
  public:

    Ui_PlyCellGraphDlg ui;

    QStringList attrMapsToBeSaved;

    SaveAttributesCSV(const Process& process) : Process(process)
    {
	  setName("Mesh/Attributes/Save to CSV");
	  setDesc("Save Measure Attribute Maps to a csv file. Labels and parent labels will always be saved in the first two columns.");
	  setIcon(QIcon(":/images/MakeHeatMap.png"));

	  addParm("File Name","Name of the file to save to (if empty Dialog will be opened)","");
	  addParm("Attribute Name","Attribute name. Comma separated list of attribute names (if empty Dialog will be opened)","");
	}

    bool processParms();

    bool initialize(QWidget* parent);

    bool run()
    {
      if(!checkState().mesh(MESH_ANY))
        return false;
      Mesh *mesh = currentMesh();
      return run(mesh, parm("File Name"));
    }

    bool run(Mesh* mesh, QString filename);

    public slots:

      void selectAll();
      void unselectAll();
  };

  class mgxBase_EXPORT SaveAttributesCSVExtended : public Process
  {
  Q_OBJECT
  public:

    Ui_PlyCellGraphDlg ui;

    QStringList attrMapsToBeSaved;

    SaveAttributesCSVExtended(const Process& process) : Process(process)
    {
	  setName("Mesh/Attributes/Save to CSV Extended");
	  setDesc("Save Measure Attribute Maps to a csv file. Labels and parent labels will always be saved in the first two columns.\n"
            "Extended process that includes extra columns to facilitate the data import into R and other platforms.");
	  setIcon(QIcon(":/images/MakeHeatMap.png"));

	  addParm("File Name","File name (if empty Dialog will be opened)","");
	  addParm("Missing Data Handling","Missing Data Handling","Empty", QStringList() << "Empty" << "0" << "NA");
	  addParm("Genotype","Add a genotype name column. Leave empty to ignore","");
	  addParm("Sample Name","Add a sample name column. Leave empty to ignore","");
	  addParm("Time Point","Add a time point column. Leave empty to ignore","");
    addParm("Stage","Add a stage column. Leave empty to ignore","");
	}

    bool processParms();

    bool initialize(QWidget* parent);

    bool run()
    {
      if(!checkState().mesh(MESH_ANY))
        return false;
      Mesh *mesh = currentMesh();
      return run(mesh, parm("File Name"), parm("Missing Data Handling"), parm("Genotype"), parm("Sample Name"), parm("Time Point"), parm("Stage"));
    }

    bool run(Mesh* mesh, QString filename, QString missingData, QString genotype, QString sample, QString timepoint, QString stage);

    void setAttrs(QStringList attrList)
    {
      attrMapsToBeSaved = attrList;
    }

    public slots:

      void selectAll();
      void unselectAll();
  };

  class mgxBase_EXPORT ClearAttrMap : public Process
  {
  public:
    ClearAttrMap(const Process& process) : Process(process)
    {
		  setName("Mesh/Attributes/Clear Attr Map");
	    setDesc("Create an empty attribute map of the specified type");
	    setIcon(QIcon(":/images/Clear.png"));

		  addParm("Type","","Double", QStringList() << "Double" << "Int" << "Vector" << "Tensor");	// 0
		  addParm("Name","","EmptyAttrMap");	// 1
	  }

    bool run()
    {
      if(!checkState().mesh(MESH_ANY))
        return false;
      Mesh *mesh = currentMesh();
      return run(mesh, parm("Type"), parm("Name"));
    }

    bool run(Mesh* mesh, QString type, QString name);

  };


}
#endif
