//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef CELLPOSE_HPP
#define CELLPOSE_HPP

#include <CellposeConfig.hpp>
#include <Process.hpp>
#include <Information.hpp>
#include <Progress.hpp>

namespace mgx
{
  ///\addtogroup Process
  ///@{
  /**
   * \class CellposeSegment Cellpose.hpp <Cellpose.hpp>
   *
   * Perform a segmentation with Cellpose 
   */
  class Cellpose_EXPORT CellposeSegment : public Process 
  {
  public:
    CellposeSegment(const Process &process) : Process(process) 
    {
      setName("Stack/CNN/Cellpose Segment");
      setDesc("Perform a segmentation with Cellpose. If you use this process, please cite: https://arxiv.org/abs/2105.00794\n");
      setIcon(QIcon(":/images/Cellpose.png"));

      addParm("Variance", "Variance of the gaussian kernel applied to input channels.", "1.0");
      addParm("Iterations", "Number of iterations (should be slightly larger than the radius of the biggest object)", "2000");
      addParm("Closing Radius", "Radius to be used to fuse nearby sinks", "10.0");
      addParm("Segment", "Return segmentation (otherwise cell boundaries)", "Yes", booleanChoice());
      addParm("Run CNN", "Run the Cellpose CNN first (No uses previous values)", "Yes", booleanChoice());
      addParm("Model Path", "Path to network model", QString("%1/CNN").arg(MGXRESPATH));
	  }
  
    bool run()
    {
      Stack* stack = currentStack();
      Store* input = currentStack()->currentStore();
      Store* output = currentStack()->work();
      return run(stack, input, output, stringToBool(parm("Segment")), stringToBool(parm("Run CNN")), parm("Model Path"));
    }
  
    bool run(Stack *stack, Store* input, Store* output, bool segment, bool runCNN, const QString &modelPath);
  };

  class Cellpose_EXPORT CellposeSegmentCells : public CellposeSegment 
  {
  public:
    CellposeSegmentCells(const Process &process) : CellposeSegment(process) 
    {
      setName("Stack/CNN/Cellpose Segment Cells");

      setParmDefault("Model Path", QString("%1/CNN/Cellpose3DCells.pt").arg(MGXRESPATH));
    }
  };

  class Cellpose_EXPORT CellposeSegmentNuclei : public CellposeSegment 
  {
  public:
    CellposeSegmentNuclei(const Process &process) : CellposeSegment(process) 
    {
      setName("Stack/CNN/Cellpose Segment Nuclei");

      setParmDefault("Model Path", QString("%1/CNN/Cellpose3DNuclei.pt").arg(MGXRESPATH));
    }
  };
  ///@}
}
#endif
