//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef COLORBAR_H
#define COLORBAR_H

#include <Config.hpp>
#include <GL.hpp>

#include <MGXViewer/qglviewer.h>
#include <Parms.hpp>

#include <iostream>
#include <QDataStream>
#include <QString>
#include <QTextStream>
#include <string>
#include <valarray>

namespace mgx 
{
  class mgx_EXPORT Colorbar 
	{
  public:
    typedef std::valarray<double> array;
    enum Position { TOP, BOTTOM, LEFT, RIGHT, TOP_LEFT, TOP_RIGHT, BOTTOM_RIGHT, BOTTOM_LEFT };
  
    enum Orientation { HORIZONTAL, VERTICAL };
  
    Colorbar(Position pos = RIGHT);
  
    void draw(GLuint colormapTexId, QPaintDevice* device) const;
  
    Position position;
    Orientation orientation;
    QFont font;
  
    double scale_length;
    double width;
    double distance_to_border;
    double text_to_bar;
    double tick_size;
    double exp_size;
    double epsilon;
    double line_width;
  
    double vmin, vmax;
  
    double globalScaling;
  
    QString label;
  
    void readParms(Parms& parms, QString section);
    void writeParms(QTextStream& pout, QString section);
  
    void scaleDrawing(double scale);
    void restoreScale();
  
  protected:
    void startScreenCoordinatesSystem(QPaintDevice* device) const;
    void stopScreenCoordinatesSystem() const;
  
    mutable double prev_width, prev_height;
    void getValues(double start, double end, double delta, array& result) const;
    array selectValuesDirect(double length, bool is_vertical, const QFontMetricsF& metric, double min_dist) const;
    std::pair<double, double> significantDigits(double start, double end) const;
    bool canRenderTicks(const array& ticks, double length, double min_dist, bool is_vertical,
                         const QFontMetricsF& font_metric) const;
    QStringList _tick2str(const array& ticks, QString* extra = 0) const;
    array selectValues(double length, bool is_vertical, const QFontMetricsF* metric = 0) const;
  };
  
  std::ostream& operator<<(std::ostream& s, const Colorbar::Position& pos);
  std::istream& operator>>(std::istream& s, Colorbar::Position& pos);
  
  QTextStream& operator<<(QTextStream& s, const Colorbar::Position& pos);
  QTextStream& operator>>(QTextStream& s, Colorbar::Position& pos);
  
  QDataStream& operator<<(QDataStream& s, const Colorbar::Position& pos);
  QDataStream& operator>>(QDataStream& s, Colorbar::Position& pos);
}
#endif
