//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef _CONTOUR_HPP_
#define _CONTOUR_HPP_
/**
 * \file contour.h
 *
 * Defines the Contour class
 */

#include <Config.hpp>
#include <vector>
#include <Vector.hpp>

namespace mgx
{
  /**
   * \class Contour contour.h <util/contour.h>
   * \brief Contour utility class.
   *
   * The Contour class encapsulates b-spline contours specified in
   * the VLAB contour formats.  Currently all versions are supported
   * (original, ver. 1 1 and 1.4).  Instances of Contour behave as
   * function objects.
   */
  class mgx_EXPORT Contour
  {
  public:
    Contour();
    Contour(const Contour&);
#ifdef USE_CXX0X
    Contour(Contour&&) = default;
#endif

    Contour(const QString &fileName);
    Contour& operator=(const Contour&);
#ifdef USE_CXX0X
    Contour& operator=(Contour&&) = default;
#endif
    Vector<3,double> operator()(double t) const;
    const Vector<3,double>& getMax() const;
    const Vector<3,double>& getMin() const;

    double length(double a, double b, double dt = 0.01);
    double travel(double t, double l, double dt = 0.01);

    Vector<3,double> tangent(double t, double dt = 0.01);
    Vector<3,double> normal(double t, double dt = 0.01);

    void reread();

  private:
    QString fileName;
    double Basis0(double t) const;
    double Basis1(double t) const;
    double Basis2(double t) const;
    double Basis3(double t) const;

    std::vector<Vector<3,double> > pts;
    Vector<3,double> max;
    Vector<3,double> min;

    bool closed;
    bool regular;
  };
}

#endif
