//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef DIVISION_ANALYSIS_2D_HPP
#define DIVISION_ANALYSIS_2D_HPP

#include <Process.hpp>
#include <DivisionAnalysisData.hpp>
#include <CellDivision.hpp>

#include <DivisionAnalysisProcesses.hpp>

namespace mgx {


 /**
   * \class FlatDivisionPlane2D DivisionAnalysis.hpp <DivisionAnalysis.hpp>
   *
   * FlatDivisionPlane2D
   * Display a planar approximation plane of a wall between two cells
   */
   class FlatDivisionPlane2D : public Process
  {
    public:
      FlatDivisionPlane2D(const Process& process) : Process(process) 
      { 
		setName("Mesh/Division Analysis/Analysis 2D/Division Plane Approximation");
		setDesc("Display a planar approximation plane of a wall between two cells");
		setIcon(QIcon(":/images/CellDivide.png"));

		addParm("Display Plane","Overwrite the non-active mesh and display the plane","Yes",booleanChoice());
    addParm("Plane Size","Plane Size","10.0");
		addParm("Reset Mesh","Reset Mesh","No",booleanChoice());
		  
	   }

       bool run()
       {
         Mesh *m = currentMesh();
         Mesh *m2;
         if(currentMesh() == mesh(0))
           m2 = mesh(1);
         else
           m2 = mesh(0);
         return run(m, m2, stringToBool(parm("Display Plane")), parm("Plane Size").toDouble(), stringToBool(parm("Reset Mesh")));
       }
       bool run(Mesh *m, Mesh *m2, bool display, double planeSize, bool resetMesh, int label = 0);

  };

 /**
   * \class TestDivisionPlanes2D DivisionAnalysis.hpp <DivisionAnalysis.hpp>
   *
   * TestDivisionPlanes2D
   * 2D process
   */
  class TestDivisionPlanes2D : public Process
  {
    public:
      TestDivisionPlanes2D(const Process& process) : Process(process) 
      { 
		setName("Mesh/Division Analysis/Analysis 2D/Division Analysis");
		setDesc("Finds the shortest wall of a cell PRE division \n"
        "(needs a single cell as input)\n"
        "The result will appear in (and overwrite) the non-active mesh.");
		setIcon(QIcon(":/images/CellDivide.png"));

		addParm("Nr of Planes","Approx number of planes to be tested","1000");
		addParm("Steps Center Displacement","Displacement steps from the cell center","0");  
		addParm("Stepsize Center Displacement","Displacement step size from the cell center","0.0");  
		addParm("Division Point","Division Point", "Centroid",QStringList() << "Centroid" << "Center Actual" << "Custom Vertex");
		  
		  }

      bool run()
      {
        Mesh *m = currentMesh();
        Mesh *m2;
        if(currentMesh() == mesh(0))
          m2 = mesh(1);
        else
          m2 = mesh(0);
        return run(m, m2, parm("Nr of Planes").toInt(), parm("Steps Center Displacement").toInt(), parm("Stepsize Center Displacement").toDouble(),
         parm("Division Point"), -1);
      }
      bool run(Mesh *m, Mesh *m2, int divPlanes, int centerDisplSteps, double centerDisplStepSize, QString divCenter, int parentLabel);

  };
 /**
   * \class DivisionAnalysisMulti2D DivisionAnalysis.hpp <DivisionAnalysis.hpp>
   *
   * DivisionAnalysisMulti2D
   * 2D process
   */
  class DivisionAnalysisMulti2D : public Process
  {
    public:
      DivisionAnalysisMulti2D(const Process& process) : Process(process) 
      { 
		setName("Mesh/Division Analysis/Analysis 2D/Division Analysis Multi");
		setDesc("Multi cell analysis of divisions in 2D meshes");
		setIcon(QIcon(":/images/CellDivide.png"));

		addParm("Test Single","Test Single","No",booleanChoice());
		addParm("Test Double","Test Double","Yes",booleanChoice());  	 		  
	  }

      bool run()
      {
        Mesh *m = currentMesh();
        Mesh *m2;
        if(currentMesh() == mesh(0))
          m2 = mesh(1);
        else
          m2 = mesh(0);
        return run(m, m2, stringToBool(parm("Test Single")), stringToBool(parm("Test Double")));
      }
      bool run(Mesh *m, Mesh *m2, bool singleCells, bool doubleCells);

  };



}

#endif
