//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef ITKPROCESS_HPP
#define ITKPROCESS_HPP

#include <ITKConfig.hpp>
#include <Process.hpp>

#include <itkObject.h>
#include <itkImage.h>
#include <itkImportImageFilter.h>
#include <itkImageSliceConstIteratorWithIndex.h>
#include <itkImageRegionConstIterator.h>
#include <itkImageFileWriter.h>
#include <itkAdaptImageFilter.h>

namespace mgx 
{
  typedef itk::Image<unsigned short int, 3> UImageType;
  typedef itk::Image<signed short int, 3> SImageType;
  typedef itk::Image<float, 3> FImageType;
  
  struct mgxITKutil_EXPORT UnsignedToSignedAccessor 
  {
    typedef signed short ExternalType;
    typedef unsigned short InternalType;
  
    static ExternalType Get(const InternalType& value) 
    {
      return ExternalType(value >> 1);
    }
  
    static void Set(InternalType& output, const ExternalType& value)
    {
      if(value > 0)
        output = InternalType(value) << 1;
      else
        output = 0;
    }
  };
  
  struct mgxITKutil_EXPORT SignedToUnsignedAccessor 
  {
    typedef unsigned short ExternalType;
    typedef signed short InternalType;
  
    static ExternalType Get(const InternalType& value)
    {
      if(value > 0)
        return ExternalType(value) << 1;
      return 0;
    }
  
    static void Set(InternalType& output, const ExternalType& value) 
    {
      output = InternalType(value >> 1);
    }
  };
  
  struct mgxITKutil_EXPORT UnsignedToFloatAccessor 
  {
    typedef float ExternalType;
    typedef unsigned short InternalType;
  
    static ExternalType Get(const InternalType& value) 
    {
      return float(value) / 65535.;
    }
  
    static void Set(InternalType& output, const ExternalType& value)
    {
      if(value > 0)
        output = InternalType(value * 65535.);
      else
        output = 0;
    }
  };
  
  struct mgxITKutil_EXPORT FloatToUnsignedAccessor 
  {
    typedef unsigned short ExternalType;
    typedef float InternalType;
  
    static ExternalType Get(const InternalType& value)
    {
      if(value > 0)
        return ExternalType(value * 65535.);
      else
        return 0u;
    }
  
    static void Set(InternalType& output, const ExternalType& value) 
    {
      output = InternalType(float(value) / 65535.);
    }
  };
  
  class mgxITKutil_EXPORT FImageConverter : public itk::Object 
  {
    FImageConverter();
  
    typedef itk::ImportImageFilter<ushort, 3> ImportFilterType;
    typedef itk::AdaptImageFilter<UImageType, FImageType, UnsignedToFloatAccessor> AdapterType;
  
  public:
    ~FImageConverter();
  
    typedef FImageConverter Self;
    typedef itk::SmartPointer<const Self> ConstPointer;
    typedef itk::SmartPointer<Self> Pointer;
  
    virtual itk::SmartPointer<itk::LightObject> CreateAnother() const;
    virtual const char* GetNameOfClass() const;
  
    void SetStore(const Store* store);
    void Update();
    FImageType::Pointer GetOutput();
  
    static bool TransferImage(HVecUS &data, FImageType::ConstPointer image);
  
    static Pointer New();
  
  protected:
    const Store* store;
    ImportFilterType::Pointer importer;
    AdapterType::Pointer adapter;
  };
  
  class mgxITKutil_EXPORT SImageConverter : public itk::Object {
    SImageConverter();
  
    typedef itk::ImportImageFilter<ushort, 3> ImportFilterType;
    typedef itk::AdaptImageFilter<UImageType, SImageType, UnsignedToSignedAccessor> AdapterType;
  
  public:
    ~SImageConverter();
  
    typedef SImageConverter Self;
    typedef itk::SmartPointer<const Self> ConstPointer;
    typedef itk::SmartPointer<Self> Pointer;
  
    virtual itk::SmartPointer<itk::LightObject> CreateAnother() const;
    virtual const char* GetNameOfClass() const;
  
    void SetStore(const Store* store);
    void Update();
    SImageType::Pointer GetOutput();
  
    static bool TransferImage(HVecUS &data, SImageType::ConstPointer image);
  
    static Pointer New();
  
  protected:
    const Store* store;
    ImportFilterType::Pointer importer;
    AdapterType::Pointer adapter;
  };
  
  class mgxITKutil_EXPORT UImageConverter : public itk::Object 
  {
    UImageConverter();
  
    typedef itk::ImportImageFilter<ushort, 3> ImportFilterType;
  
  public:
    ~UImageConverter();
  
    typedef UImageConverter Self;
    typedef itk::SmartPointer<const Self> ConstPointer;
    typedef itk::SmartPointer<Self> Pointer;
    typedef ImportFilterType::SpacingType SpacingType;
    typedef ImportFilterType::RegionType RegionType;
    typedef ImportFilterType::OriginType OriginType;
    typedef itk::Vector<double, 3> itkPoint3d;
  
    virtual itk::SmartPointer<itk::LightObject> CreateAnother() const;
    virtual const char* GetNameOfClass() const;
  
    void Update();
    UImageType::Pointer GetOutput();
  
    const SpacingType& GetSpacing() const { return importer->GetSpacing(); }
    const RegionType& GetRegion() const { return importer->GetRegion(); }
    const OriginType& GetOrigin() const { return importer->GetOrigin(); }

    bool SetStore(const Store* store);
    bool SetData(const HVecUS &data, const Point3u &size);

    void SetSpacing(const Point3f &spacing) { double s[3] = {spacing.x(), spacing.y(), spacing.z()}; importer->SetSpacing(s); }
    void SetOrigin(const Point3f &origin) { double o[3] = {origin.x(), origin.y(), origin.z()}; importer->SetOrigin(o); }
  
    static bool TransferImage(HVecUS &data, UImageType::ConstPointer image);

    static Pointer New();
  
  protected:
    ImportFilterType::Pointer importer;
  };
}

#endif
