//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef MESH_PROCESS_CREATION_HPP
#define MESH_PROCESS_CREATION_HPP

#include <Process.hpp>

#include <Polygonizer.hpp>

namespace mgx
{
  ///\addtogroup MeshProcess
  ///@{
  /**
   * \class MarchingCubeSurface ProcessCreation.hpp <MeshProcessCreation.hpp>
   *
   * Find the surface of a volume defined by an iso-surface on the image
   * intensity using a variation on the Marching Cube method.
   */
  class mgxBase_EXPORT MarchingCubeSurface : public Process, public ImplicitFunction 
  {
  public:
    MarchingCubeSurface(const Process& process) : Process(process) 
    {
		
      setName("Mesh/Creation/Marching Cubes Surface");
      setDesc("Extract surface mesh with marching cubes. A threshold to 0 will interpret the image as binary.");
      setIcon(QIcon(":/images/MarchCubes.png"));  
      addParm("Cube size (µm)","Size for the marching cubes.","5.0");		// 0
      addParm("Threshold","Minimal signal used for surface extraction.","5000");		// 1		
		
		}
  
    bool run()
    {
      if(!checkState().store(STORE_NON_EMPTY).mesh())
        return false;
  
      _stack = currentStack();
      _store = _stack->currentStore();
      Mesh* mesh = currentMesh();
  
      bool result = run(mesh, _store, parm("Cube size (µm)").toFloat(), parm("Threshold").toInt());
      if(result)
        mesh->setShowVertex("Signal");
      if(result and not mesh->showSurface() and not mesh->showMesh())
        mesh->setShowSurface(true);
      return result;
    }
  
    bool run(Mesh* mesh, const Store* store, float cubeSize, int threshold);
 
    bool eval(Point3f p);
  
  private:
    int _threshold;
    const Stack* _stack;
    const Store* _store;
  };
  
  /**
   * \class MarchingCube3D ProcessCreation.hpp <MeshProcessCreation.hpp>
   *
   * Find the surface of the segmented volumes using a variation on
   * the Marching Cube method.
   */
  class mgxBase_EXPORT MarchingCube3D : public Process, public ImplicitFunction 
  {
  public:
    MarchingCube3D(const Process& process) : Process(process) 
    {		
      setName("Mesh/Creation/Marching Cubes 3D");
      setDesc("Extract 3D meshes with marching cubes");
      setIcon(QIcon(":/images/MarchCubes3D.png"));
      
      addParm("Cube size (µm)","Size for the marching cubes.","5.0");		// 0
      addParm("Min Voxels","Minimal number of voxels for extracting surface.","0");		// 1
      addParm("Smooth Passes","Smooth Passes.","3");		// 2
      addParm("Label","Extract surface only for this label.","0");		// 3
      addParm("Ignore Stack Type","Ignore Stack Type","No",booleanChoice());    // 3      	
	}
  
    bool run()
    {
      bool ignoreCheck = stringToBool(parm("Ignore Stack Type"));
      if(!ignoreCheck)
        if(!checkState().store(STORE_LABEL).mesh())
          return false;
  
      _stack = currentStack();
      _store = _stack->currentStore();
      Mesh* mesh = currentMesh();
  
      bool result = run(mesh, _store, parm("Cube size (µm)").toFloat(), parm("Min Voxels").toInt(), 
                                                parm("Smooth Passes").toInt(), parm("Label").toInt());
      if(result)
        mesh->setShowLabel();
      if(result and not mesh->showSurface() and not currentMesh()->showMesh())
        currentMesh()->setShowSurface(true);
      return result;
    }
  
    bool run(Mesh* mesh, const Store* store, float cubeSize, int minVoxels, 
                                                             int smooth, int singleLabel);
  
    bool eval(Point3f p);
  
  private:
    int _label;
    const Stack* _stack;
    const Store* _store;
  };
  
  /**
   * \class CuttingSurfMesh ProcessCreation.hpp <MeshProcessCreation.hpp>
   *
   * Create a surface matching the current cutting surface, whether it's a
   * plane or a Bezier surface.
   */
  class mgxBase_EXPORT CuttingSurfMesh : public Process 
  {
  public:
    CuttingSurfMesh(const Process& process) : Process(process) 
    {
      setName("Mesh/Creation/Mesh Cutting Surface");
      setDesc("Make mesh from cutting surface");				
	}
  
    bool run()
    {
      if(!checkState().mesh())
        return false;
      return run(currentMesh());
    }
  
    bool run(Mesh* mesh);
  };

  /**
   * \class VoxelFaceMesh ProcessCreation.hpp <MeshProcessCreation.hpp>
   *
   * Create a mesh by extracting the faces from the Voxels.
   */
  class mgxBase_EXPORT VoxelFaceMesh : public Process 
  {
  public:
    VoxelFaceMesh(const Process& process) : Process(process) 
    {		
      setName("Mesh/Creation/Voxel Face Mesh");
      setDesc("Extract a mesh from the faces of the voxels.");				
    }
  
    bool run()
    {
      if(!checkState().mesh().store(STORE_LABEL))
        return false;
      return run(currentStack(), currentStack()->currentStore(), currentMesh());
    }
  
    bool run(Stack *stack, Store *store, Mesh *mesh);
  };


  class mgxBase_EXPORT MeshFromLocalMaxima : public Process {
  public:
    MeshFromLocalMaxima(const Process& process) : Process(process) 
    {
      setName("Mesh/Creation/Mesh From Local Maxima");
      setDesc("Create a Mesh from Local Maxima");

      addParm("Radius (µm)","Radius (µm)","3.0");	
      
    }			
						
    bool run()
    {
      if(!checkState().store(STORE_NON_EMPTY))
        return false;
      return run(currentMesh(), currentStack()->currentStore(), parm("Radius (µm)").toFloat());
    }
  
    bool run(Mesh* mesh, const Store* store, float radius);
  };

class mgxBase_EXPORT CreateSmoothSurface : public Process {
  public:
    CreateSmoothSurface(const Process& process) : Process(process) 
    {
      setName("Mesh/Creation/Surface From 3D Labels/A Create Smooth Surface");
      setDesc("Create a smooth surface mesh from an edge detect result.\n"
      " This process requires an edge detect result in the active stack.\n"
      " The following sub-processes are then executed (in this order):\n"
        "- Gaussian Blur Stack\n"
        "- Marching Cubes Surface\n"
        "- Smooth Mesh\n"
        "- Save Mesh");
      
      addParm("Blur: X Sigma (µm)","X Sigma (µm)","3");
      addParm("Blur: Y Sigma (µm)","Y Sigma (µm)","3");
      addParm("Blur: Z Sigma (µm)","Z Sigma (µm)","3"); 

      addParm("Surface: Cube size (µm)","Size for the marching cubes.","1.0");
      addParm("Surface: Threshold","Minimal signal used for surface extraction.","10000"); 

      addParm("Smooth: Passes","Passes","5");
      addParm("Smooth: Walls Only","Walls Only","No",booleanChoice());

      addParm("Save: Filename","Filename","autosave1.mgxm");    // 0
      addParm("Save: Transform","Transform","No", booleanChoice());   // 1
      addParm("Save: Mesh Number","Mesh number","0", QStringList() << "0" << "1");    // 2
      
    }     
            
    bool run()
    {
      if(!checkState().store(STORE_NON_EMPTY))
        return false;
      return run(currentMesh(), currentStack()->currentStore());
    }
  
    bool run(Mesh* mesh, const Store* store);
  };


class mgxBase_EXPORT CreateRoughSurface : public Process {
  public:
    CreateRoughSurface(const Process& process) : Process(process) 
    {
      setName("Mesh/Creation/Surface From 3D Labels/Misc/Create Rough Surface");
      setDesc("Create a rough surface mesh from an edge detect result.\n"
      " This process requires an edge detect result in the active stack.\n"
      " The following sub-processes are then executed (in this order):\n"
        "- Marching Cubes Surface\n"
        "- Smooth Mesh\n"
        "- Save Mesh");

      addParm("Surface: Cube size (µm)","Size for the marching cubes.","1.0");
      addParm("Surface: Threshold","Minimal signal used for surface extraction.","0"); 

      addParm("Smooth: Passes","Passes","5");
      addParm("Smooth: Walls Only","Walls Only","No",booleanChoice());

      addParm("Save: Filename","Filename","autosave1.mgxm");    // 0
      addParm("Save: Transform","Transform","No", booleanChoice());   // 1
      addParm("Save: Mesh Number","Mesh number","0", QStringList() << "0" << "1");    // 2
      
    }     
            
    bool run()
    {
      if(!checkState().store(STORE_NON_EMPTY))
        return false;
      return run(currentMesh(), currentStack()->currentStore());
    }
  
    bool run(Mesh* mesh, const Store* store);
  };



  class mgxBase_EXPORT Copy3DLabelsToSurface : public Process {
  public:
    Copy3DLabelsToSurface(const Process& process) : Process(process) 
    {
      setName("Mesh/Creation/Surface From 3D Labels/B Copy 3D Labels to Surface");
      setDesc("Copy 3D Labels to Surface.\n"
      " This process requires a 3D segmented stack in the active stack and a surface mesh.\n"
      " The following sub-processes are then executed (in this order):\n"
        "- Subdivide Mesh\n"
        "- Project 3D Labels on Mesh\n"
        "- Save Mesh");

      addParm("Project: Min Dist (µm)","Distance (triangle-voxel) above which the signal is projected.","0.0");  // 1
      addParm("Project: Max Dist (µm)","Maximal distance (triangle-voxel) used for signal projection.","4.0"); // 2
      addParm("Project: Labeling Method","Labeling Method","Majority",QStringList() << "Majority" << "Absolute Majority"); // 2
      addParm("Project: Ignore Zero","Ignore Zero","No",booleanChoice()); // 2

      addParm("Save: Filename","Filename","autosave1.mgxm");    // 0
      addParm("Save: Transform","Transform","No", booleanChoice());   // 1
      addParm("Save: Mesh Number","Mesh number","0", QStringList() << "0" << "1");    // 2
      
    }     
            
    bool run()
    {
      if(!checkState().store(STORE_NON_EMPTY))
        return false;
      return run(currentMesh(), currentStack()->currentStore());
    }
  
    bool run(Mesh* mesh, const Store* store);
  };


  class mgxBase_EXPORT OptimizeSurfaceSegmentation : public Process {
  public:
    OptimizeSurfaceSegmentation(const Process& process) : Process(process) 
    {
      setName("Mesh/Creation/Surface From 3D Labels/C Optimize Surface Segmentation");
      setDesc("Optimize surface mesh from 3D labels.\n"
      " This process requires a cell wall marker stack in the active stack (e.g. raw image or boundary predictions) and a surface mesh with projected 3D labels.\n"
      " The following sub-processes are then executed (in this order):\n"
        "- Subdivide Adaptive Near Borders\n"
        "- Smooth Mesh\n"
        "- Project Signal\n"
        "- Gaussian Blur Signal\n"
        "- Watershed Segmentation\n"
        "- Save Mesh");

      addParm("Subdivide: Max Area(µm²)","Area threshold (in square microns) for subdivision, triangles smaller than this won't be subdivided","0.1");
      addParm("Subdivide: Border Dist(µm)","Distance (in microns) from cell borders that triangles will be subdivided","1.0");


      addParm("Smoothing: Passes","Passes","5");
      addParm("Smoothing: Walls Only","Walls Only","No",booleanChoice());

      addParm("Project Signal: Use absolute","Use absolute values of signal, instead of normalizing it, useful for signal quantification.","No",booleanChoice());
      addParm("Project Signal: Min Dist (µm)","Distance (triangle-voxel) above which the signal is projected.","2.0");  // 1
      addParm("Project Signal: Max Dist (µm)","Maximal distance (triangle-voxel) used for signal projection.","4.0"); // 2
      addParm("Project Signal: Labeling Method","Labeling Method","Majority",QStringList() << "Majority" << "Absolute Majority"); // 2
      addParm("Project Signal: Min Signal","Lower bound of signal value if 'Use absolute' is chosen","0.0");
      addParm("Project Signal: Max Signal","Upper bound of projected signal value.","60000.0");
      addParm("Project Signal: Max Project","Use max projection instead of average","No", booleanChoice()); 

      addParm("Blur Signal: Radius (µm)","Size of neighborhood used for Gaussian blur. The blur function standard deviation is given by sigma = radius/2. ","2.0");  

      addParm("Watershed: Steps","Number of propagation steps performed before updating display. Increase value to run faster.","50000");  

      addParm("Save: Filename","Filename","autosave2.mgxm");    // 0
      addParm("Save: Transform","Transform","No", booleanChoice());   // 1
      addParm("Save: Mesh Number","Mesh number","0", QStringList() << "0" << "1");    // 2

    }     
            
    bool run()
    {
      if(!checkState().store(STORE_NON_EMPTY))
        return false;
      return run(currentMesh(), currentStack()->currentStore(), parm("Radius (µm)").toFloat());
    }
  
    bool run(Mesh* mesh, const Store* store, float radius);
  };


  ///@}
}

#endif
