//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef COORDX_PROCESSES_HPP
#define COORDX_PROCESSES_HPP

#include <CellAtlas.hpp>
#include <MeshProcessLineage.hpp>
#include <BezierProcess.hpp>
#include <MeshProcessMeasures3D.hpp>

namespace mgx 
{
  class CellAtlas_EXPORT LabelOvules : public Process
  {
  public:
    LabelOvules(const Process& process) : Process(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/C Label Ovules");
      setDesc("Select a cell at the tip of each organ to assign them organ specific cell types. The result is saved in the specified attribute map");
      setIcon(QIcon(":/images/CellAtlas.png"));

      addParm("Max Size","Max Size","6");
      addParm("Export to Attr Map","Export to Attr Map","Ovule");
    }

    bool run() {
      Mesh *m1 = currentMesh();
      return run(m1, parm("Max Size").toInt(), parm("Export to Attr Map"));
    }

    bool run(Mesh *m1, int maxSize, QString ovuleAttr);
  };

  class CellAtlas_EXPORT LabelCellsFromHeat : public Process
  {
  public:
    LabelCellsFromHeat(const Process& process) : Process(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/Tools/Label Cells From Heat");
      setDesc("Detect cell layers and label them accordingly");
      setIcon(QIcon(":/images/CellAtlas.png"));

      addParm("Heat Value 1","Heat Value 1","3");
      addParm("Heat Value 2","Heat Value 2","4");
      addParm("Heat Value 3","Heat Value 3","-1");
      addParm("Heat Value 4","Heat Value 4","-1");
      addParm("Heat Value 5","Heat Value 5","-1");
      addParm("Heat Value 6 or Greater","Heat Value 6","7");
      addParm("Overwrite Existing Parents","Overwrite Existing Parents","No",booleanChoice());
    }

    bool run() 
    {
      Mesh *m1 = currentMesh();
      return run(m1, parm("Heat Value 1").toInt(), parm("Heat Value 2").toInt(), parm("Heat Value 3").toInt(),
          parm("Heat Value 4").toInt(), parm("Heat Value 5").toInt(), parm("Heat Value 6 or Greater").toInt(), 
          stringToBool(parm("Overwrite Existing Parents")));
    }
    bool run(Mesh *m1, int h1, int h2, int h3, int h4, int h5, int h6, bool overwrite);
  };

  class CellAtlas_EXPORT CoordX_DetectCellLayers : public Process
  {
  public:
    CoordX_DetectCellLayers(const Process& process) : Process(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/B Detect Cell Layers");
      setDesc("Detect cell layers using a loaded segmented stack and label them accordingly");
      setIcon(QIcon(":/images/CellAtlas.png"));

      addParm("Number of Layers","Number of Layers","3");
      addParm("Use Stack","Use Stack (If No, only use Mesh)","Yes",booleanChoice());
      addParm("Min Shared Voxels (Stack)","Min Shared Voxels","1");
      addParm("Min Outside Wall Ratio (Mesh)","Min Outside Wall Ratio","0.35");
      addParm("Export to Attr Map","Export to Attr Map","Layers");
    }

    bool run() 
    {
      Mesh *m1 = mesh(0);
      if(stringToBool(parm("Use Stack"))){
        Stack* s1 = currentStack();
        Store* store1 = s1->currentStore();
        return run(m1, s1, store1, parm("Number of Layers").toInt(), parm("Min Shared Voxels (Stack)").toInt(), parm("Export to Attr Map"));
      } else
        return run(m1,parm("Number of Layers").toInt(), parm("Min Outside Wall Ratio (Mesh)").toDouble(), parm("Export to Attr Map"));
    }

    bool run(Mesh *m1, Stack* s1, Store* store1, int layers, int minShared, QString exportAttr);
    bool run(Mesh *m1, int layers, double minShared, QString exportAttr);
  };

  class CellAtlas_EXPORT CoordX_DetectCellLayersOvule : public CoordX_DetectCellLayers
  {
  public:
    CoordX_DetectCellLayersOvule(const Process& process) : CoordX_DetectCellLayers(process)
    {
      setName("Mesh/Cell Atlas 3D/Ovule/Detect Cell Layers");
    }
  };

  class CellAtlas_EXPORT NeighborVoxelsStack : public Process
  {
  public:
    NeighborVoxelsStack(const Process& process) : Process(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/Tools/Neighbor Voxels Stack");
      setDesc("Count for each cell how many voxels in the segmented stack are located at the outside\n"
              "(with a 0 neighbor voxel) and make a heatmap on the mesh");
      setIcon(QIcon(":/images/CellAtlas.png"));

      addParm("Heat Map","Heat Map","Outside Voxels", QStringList() << "Outside Voxels");
    }

    bool run() 
    {
      Mesh *m1 = mesh(0);
      Stack* s1 = currentStack();
      Store* store1 = s1->currentStore();
      return run(m1, s1, store1, parm("Heat Map"));
    }
    bool run(Mesh *m1, Stack* s1, Store* store1, QString heatmap);
  };

//  class CellAtlas_EXPORT RemoveRegressionHeatmap : public Process
//  {
//  public:
//    RemoveRegressionHeatmap(const Process& process) : Process(process)
//    {
//      setName("Mesh/Cell Atlas 3D/CoordX/Xtra/Remove Regression Heatmap");
//      setDesc("TBD");
//      setIcon(QIcon(":/images/CellAtlas.png"));
//
//      addParm("Heatmap X","Heatmap X","X");
//      addParm("Heatmap Y","Heatmap Y","Y");
//    }
//
//    bool run() 
//    {
//      Mesh *m1 = mesh(0);
//      //Stack* s1 = currentStack();
//      //Store* store1 = s1->currentStore();
//      return run(m1, parm("Heatmap X"), parm("Heatmap Y"));
//    }
//    bool run(Mesh *m, QString heatX, QString heatY);
//  };
//
//  class CellAtlas_EXPORT RemoveRegressionStack : public Process
//  {
//  public:
//    RemoveRegressionStack(const Process& process) : Process(process)
//    {
//      setName("Mesh/Cell Atlas 3D/CoordX/Xtra/Remove Regression Stack");
//      setDesc("TBD");
//      setIcon(QIcon(":/images/CellAtlas.png"));
//
//      addParm("Dimension","Dimension","Z");
//      addParm("Low Threshold (%)","Ignore voxels with a value lower than this","1.0");
//      addParm("High Threshold (%)","Ignore voxels with a value higher than this","99.0");
//    }
//
//    bool run() 
//    {
//      Mesh *m1 = mesh(0);
//      Stack* s1 = currentStack();
//      Store* store1 = s1->currentStore();
//      Store* store2 = s1->work();
//      return run(m1, s1, store1, store2, parm("Dimension"), parm("Low Threshold (%)").toDouble(), parm("High Threshold (%)").toDouble());
//    }
//    bool run(Mesh *m, Stack* s1, Store* store1, Store* store2, QString dim, double lowT, double highT);
//  };

  class CellAtlas_EXPORT DeleteStackLabelsByParent : public Process
  {
  public:
    DeleteStackLabelsByParent(const Process& process) : Process(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/Tools/Delete Stack Labels by Parent");
      setDesc("Delete stack labels by a given parent label");
      setIcon(QIcon(":/images/CellAtlas.png"));

      addParm("Parent Label","Parent Label","0");
    }

    bool run() 
    {
      Mesh *m1 = mesh(0);
      Stack* s1 = currentStack();
      Store* store1 = s1->currentStore();
      return run(m1, s1, store1, parm("Parent Label").toInt());
    }
    bool run(Mesh *m, Stack* s1, Store* store1, int parentLabel);
  };

//  class CellAtlas_EXPORT DefineML : public Process
//  {
//  public:
//    DefineML(const Process& process) : Process(process)
//    {
//      setName("Mesh/Cell Atlas 3D/CoordX/Xtra/DefineML");
//      setDesc("TBD");
//      setIcon(QIcon(":/images/CellAtlas.png"));
//
//      addParm("Use Surface Dirs","Use Surface Dirs","No",booleanChoice());
//      addParm("Custom Dir","Custom Dir","X",QStringList() << "X" << "Y" << "Z");
//      addParm("Weight Surface Dirs (%)","Weight Use Surface Dirs (%)","50.0");
//    }
//
//    bool run() 
//    {
//      Mesh *m1 = currentMesh();
//      Mesh *m2 = otherMesh();
//      return run(m1, m2, parm("Custom Dir"), stringToBool(parm("Use Surface Dirs")), parm("Weight Surface Dirs (%)").toDouble());
//    }
//    bool run(Mesh *m1, Mesh *m2, QString customDir, bool useSurf, double weightSurf);
//  };
//
//
//  class CellAtlas_EXPORT DefineAntPost : public Process
//  {
//  public:
//    DefineAntPost(const Process& process) : Process(process)
//    {
//      setName("Mesh/Cell Atlas 3D/CoordX/Xtra/DefineAntPost");
//      setDesc("TBD");
//      setIcon(QIcon(":/images/CellAtlas.png"));
//
//      addParm("Separation","Separation","Mean",QStringList() << "Mean" << "Median");
//    }
//
//    bool run() 
//    {
//      Mesh *m1 = currentMesh();
//      return run(m1, parm("Separation"));
//    }
//    bool run(Mesh *m1, QString sep);
//  };

  class CellAtlas_EXPORT SelectCellCutByBezier : public Process
  {
  public:
    SelectCellCutByBezier(const Process& process) : Process(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/Tools/Select Cells Cut By Bezier");
      setDesc("Select all cells that are cut by the current Bezier grid.");
      setIcon(QIcon(":/images/CellAtlas.png"));
    }

    bool run() 
    {
      Mesh *m1 = currentMesh();
      Stack *s1 = currentStack();
      return run(s1, m1);
    }
    bool run(const Stack *s1, Mesh *m1);
  };

  class CellAtlas_EXPORT CoordX_CellAnalysis3D : public CellAnalysis3D
  {
  public:
    CoordX_CellAnalysis3D(const Process& process) : CellAnalysis3D(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/A Cell Analysis 3D");
    }
  };

  class CellAtlas_EXPORT CoordX_UniqueParentsFromAttr : public UniqueParentsFromAttr
  {
  public:
    CoordX_UniqueParentsFromAttr(const Process& process) : UniqueParentsFromAttr(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/D Unique Parents");
    }
  };

  class CellAtlas_EXPORT CoordX_NewBezierRingPoint : public NewBezierRing 
  {
  public:
    CoordX_NewBezierRingPoint(const Process& process) : NewBezierRing(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/Generate Bezier Point Origin");
      setDesc("Creates a tiny point-like Bezier ring.");

      addParm("Radius X","Physical size along X (um)","2"); // 2
      addParm("Radius Y","Physical size along Y (um)","2"); // 3
    }
  };

  class CellAtlas_EXPORT CoordX_NewBezierRing : public NewBezierRing 
  {
  public:
    CoordX_NewBezierRing(const Process& process) : NewBezierRing(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/Generate Bezier Ring Origin");
      setDesc("New Bezier ring from parameters.");
    }
  };


  class CellAtlas_EXPORT CoordX_Measure3DCellDistanceBezierRing : public Measure3DCellDistanceBezierRing 
  {
  public:
    CoordX_Measure3DCellDistanceBezierRing(const Process& process) : Measure3DCellDistanceBezierRing(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/Measure 3D CoordX");
    }
  };


  class CellAtlas_EXPORT CoordX_Measure3DCellDistance : public Measure3DCellDistance 
  {
  public:
    CoordX_Measure3DCellDistance(const Process& process) : Measure3DCellDistance(process)
    {
      setName("Mesh/Cell Atlas 3D/CoordX/Measure 3D Cell Distance");
    }
  };
}

#endif
