//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef HASH_HPP
#define HASH_HPP

#include <QtGlobal>
#include <Config.hpp>
#include <Vertex.hpp>
#include <QString>
#include <utility>
#include <functional>
#include <boost/functional/hash.hpp>

// Add hash for unordered maps
namespace std 
{
  /** Hash pairs */
  template <typename T1, typename T2>
  struct hash<std::pair<T1, T2> >
  {
    size_t operator()(const std::pair<T1, T2> &v) const
    {
      std::size_t seed = 0;
      boost::hash_combine(seed, v.first);
      boost::hash_combine(seed, v.second);
      return seed;
    }
  };

  /** Hash vertex/cell pairs */
  template <typename T1, typename T2>
  struct hash<std::pair< mgx::Vertex<T1>, mgx::Vertex<T2> > >
  {
    size_t operator()(const std::pair<mgx::Vertex<T1>, mgx::Vertex<T2> > &v) const
    {
      std::size_t seed = 0;
      boost::hash_combine(seed, v.first.id());
      boost::hash_combine(seed, v.second.id());
      return seed;
    }
  };

  // Hash QString
  // BJL 2021-04-19: As of Qt 5.14, Qt itself defines std::hash<QString>.
#if QT_VERSION < QT_VERSION_CHECK(5, 14, 0)
  template <> struct hash<QString>
  {

    size_t operator()(const QString &s) const
    {
      int hash = 0, strlen = s.length(), i;
      QChar character;
      if (strlen == 0)
        return hash;
      for (i = 0; i < strlen; i++) {
        character = s.at(i);
        hash = (31 * hash) + (character.unicode());
      }
      return hash;
    }
  };
#endif // Qt version < 5.14
}
#endif
