//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef ITKSEGMENTATION_HPP
#define ITKSEGMENTATION_HPP

#include <ITKConfig.hpp>
#include <ITKProcess.hpp>

namespace mgx 
{
  class mgxITK_EXPORT ITKWatershed : public Process 
  {
  public:
    ITKWatershed(const Process& process) : Process(process) 
    {
      setName("Stack/ITK/Segmentation/ITK Watershed");
      setDesc("Seeded watershed segmentation, image in main store, labels in work");
      setIcon(QIcon(":/images/SegmentMesh.png"));

      addParm("Fully Connected", "Fully connected", "No", booleanChoice());
      addParm("Mark Watershed Line", "Mark Watershed Line", "No", booleanChoice());	
	  }
  
    bool run()
    {
      if(!checkState().store(STORE_MAIN | STORE_NON_LABEL) .store(STORE_WORK | STORE_LABEL))
        return false;
      Stack* stack = currentStack();
      Store* main = stack->main();
      Store* labels = stack->work();
      bool connect8 = stringToBool(parm("Fully connected"));
      bool markLine = stringToBool(parm("Mark Watershed Line"));
      return run(main, labels, connect8, markLine);
    }
  
    bool run(const Store* image, Store* labels, bool connect8, bool markLine);
  };
  
  class mgxITK_EXPORT ITKWatershedAutoSeeded : public Process 
  {
  public:
    ITKWatershedAutoSeeded(const Process& process) : Process(process) 
    {
	    setName("Stack/ITK/Segmentation/ITK Watershed Auto Seeded");
	    setDesc("Auto Seeded Watershed segmentation");
	    setIcon(QIcon(":/images/SegmentMesh.png"));

	    addParm("Fully connected","Fully connected","No",booleanChoice());
	    addParm("Mark Watershed Line","Mark Watershed Line","No",booleanChoice());
	    addParm("Level","Level","1500");						
	  }
  
    bool run()
    {
      if(!checkState().store(STORE_VISIBLE | STORE_NON_LABEL))
        return false;
      Stack* stack = currentStack();
      Store* main = stack->currentStore();
      Store* labels = stack->work();
      bool connect8 = stringToBool(parm("Fully connected"));
      bool markLine = stringToBool(parm("Mark Watershed Line"));
      if(run(stack, main, labels, parm("Level").toUInt(), connect8, markLine)) {
        main->hide();
        labels->show();
        return true;
      }
      return false;
    }
  
    bool run(Stack* stack, const Store* image, Store* labels, ushort level, bool connect8, bool markLine);
  
  };

  class mgxITK_EXPORT ITKWatershedResegment : public Process 
  {
  public:
    ITKWatershedResegment(const Process& process) : Process(process) 
    {
	    setName("Stack/ITK/Segmentation/ITK Watershed Resegment");
	    setDesc("Re-segment current label with the ITK auto seeded watershed");
	    setIcon(QIcon(":/images/SegmentMesh.png"));

	    addParm("Fully connected","Fully connected","No",booleanChoice());
	    addParm("Mark Watershed Line","Mark Watershed Line","No",booleanChoice());
	    addParm("Level","Level","1000");						
	  }
  
    bool run()
    {
      Stack *stack = currentStack();
      Store *main = stack->main();
      Store *work = stack->work();
      if(main->labels())
        throw QString("%1:run Main stack should not be labeled").arg(name());
      if(!work->labels())
        throw QString("%1:run Work stack should be labeled").arg(name());
      bool connect8 = stringToBool(parm("Fully connected"));
      bool markLine = stringToBool(parm("Mark Watershed Line"));
      return run(stack, main, work, selectedLabel(), parm("Level").toUInt(), connect8, markLine);
    }
  
    bool run(Stack* stack, const Store* image, Store* labels, int label, ushort level, bool connect8, bool markLine);
  };
}

#endif
