//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef IMAGE_HPP
#define IMAGE_HPP

#include <Config.hpp>
#include <Types.hpp>
#include <Information.hpp>

#include <cuda/CudaExport.hpp>
#include <Vector.hpp>

#include <QString>
#include <QStringList>

#include <tiffio.h>
#include <tiff.h>

namespace mgx 
{
  class Progress;
  
  typedef Vector<3, unsigned int> Point3u;
  
  class mgx_EXPORT Image3D 
  {
  public:
    Image3D() {}
    Image3D(const Point3u &size, const Point3f &step = Point3f(1, 1, 1), bool labels = false, 
        int precision = 16, int channels = 1, bool isSigned = false, bool isFloat = false)
      : size(size), step(step), labels(labels), precision(precision), channels(channels), isSigned(isSigned), isFloat(isFloat) {}
    ~Image3D() {}

    size_t totalSize();
    size_t resize();
    int bytesPerVoxel();

    bool checkDataSize(const QString &msg);
    bool calcVoxel(uchar *src, uchar *dst);
    float calcRange();
    bool toHVecUS(HVecUS &hvec);
    bool fromHVecUS(const HVecUS &vec);
    bool fromFloatVec(const FloatVec &vec);
    bool loadTiff(const QString &filename);
    bool saveTiff(const QString &filename);

    std::vector<uchar> data;

    Point3u size;
    Point3f step;
    bool labels = false;   ///< If the image is a labeled stack
    int precision = 16;    ///< size of the voxel components in bits
    int channels = 1;      ///< number of channels per pixel
    bool isSigned = false;
    bool isFloat = false;

    float minValue = 0;    ///< Min value when converted to single channel, used for scaling
    float maxValue = 0;    ///< Max value when converted to single channel, used for scaling

  private:
    // Class to handle steps for different orientations
    struct StripDataIncrement 
    {
      int startRow;
      int startColumn;
      int shiftRow;
      int shiftColumn;
      int lineEnd;
    };
    StripDataIncrement incr;

    bool readTiffInfo(TIFF *tif);
    bool readStrip(uchar *&p, uchar *strip, size_t nPixels);
  };

  struct TIFFHandler 
  {
    typedef TIFF* pointer;

    TIFFHandler(TIFF* t) : tif(t) {}

    ~TIFFHandler()
    {
      if(tif)
        TIFFClose(tif);
      tif = 0;
    }
  
    operator bool() const { return bool(tif); }
  
    operator pointer() {
      return tif;
    }
  
    TIFF* tif;
  };
  
  inline void tiffErrorHandler(const char* module, const char* fmt, va_list ap)
  {
    Information::out << "TIFF Error in module " << module << endl;
    vfprintf(stderr, fmt, ap);
  }
  
  inline void tiffWarningHandler(const char* module, const char* fmt, va_list ap)
  {
    Information::out << "TIFF warning in module " << module << endl;
    vfprintf(stderr, fmt, ap);
  } 

  // In case of an error, this function will throw a QString with the description of the error
  //mgx_EXPORT bool saveImage(QString filename, const Image3D &data, QString type = "CImg Auto", uint nDigits = 0);
;

  mgx_EXPORT HVecUS resize(const HVecUS& data, const Point3i& before, const Point3i& after, bool center);
}

#endif
