//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef LOBYNESS3D_HPP
#define LOBYNESS3D_HPP

#include <Process.hpp>
#include <MeshProcessLineage.hpp>
#include <BezierProcess.hpp>
#include <MeshProcessMeasures3D.hpp>

namespace mgx 
{ 
  // Extract the convex hull of a mesh
  class ConvexHullCGAL : public Process
  {
  public:
    ConvexHullCGAL(const Process& process) : Process(process) 
    {
      setName("Mesh/Cell Atlas 3D/CoordX/Tools/Convex Hull From Mesh"); // FIXME Why is this here?
      setDesc("Convex Hull From Mesh\n"
        "Creates the convex hull from the current mesh &\n"
        "saves the result in the non-active mesh\n"
        "(uses the CGAL library)\n");
      setIcon(QIcon(":/images/CHull.png"));

      addParm("Work on Selection", "Work on Selection", "No", booleanChoice());
    } 

    bool run()
    {
      Mesh *mesh = currentMesh();
      if(!mesh) 
        throw QString("%1 No current mesh").arg(name());

      Mesh *mesh2 = otherMesh();
      
      // consider rotation of stack
      Stack *s = currentStack();
      Matrix4d rotMatrixS1;
      s->getFrame().getMatrix(rotMatrixS1.data());
      Matrix4d mGLTot = transpose(rotMatrixS1);
      
      bool result = run(mesh, mesh2, stringToBool(parm("Work on Selection")), mGLTot);

      return result;
    }
    bool run(Mesh* mesh, Mesh* mesh2, bool workOnSel, Matrix4d mGLTot);
  };


 /**
   * \class Lobyness3D DivisionAnalysis.hpp <DivisionAnalysis.hpp>
   *
   * Lobyness3D
   */
  class Lobyness3D : public Process
  {
    public:
      Lobyness3D(const Process& process) : Process(process){

      	setName("Mesh/Shape Analysis/Lobyness Measures 3D");
		setDesc("Lobyness Plugin 3D Version\n"
        "This plugin provides several measures to analyse cell-outlines.\n"
        "Convex (Surface Area): Ratio of cell-surface area over that of its convex hull. Takes a value of 1 for convex shapes, and increases with the complexity of the volume.\n"
        "Convex (Volume): Like Convex (Surface Area), but uses volumes instead of surface areas.\n"
        "Sphericity: (6*cell-volume)^2/3 * PI^1/3 / (cell-area).\n");
		setIcon(QIcon(":/images/CHull.png"));

		addParm("Measure","Measure","Convexity",QStringList() << "Convexity" <<"Solidity"<<"Sphericity"<<"Compactness");
      }

      bool run()
     {
      QString measure = parm("Measure");
      Mesh *m = currentMesh();
      return run(m, measure);
     }

      bool run(Mesh *m, const QString &measure);

  };


class Lobyness3DConvexSurfArea : public Process
  {
  public:
    Lobyness3DConvexSurfArea(const Process& process) : Process(process) {
      setName("Mesh/Heat Map/Measures 3D/Lobeyness/Convexity");
		  setDesc("Lobyness Plugin 3D Version\n"
                "Convexity: Cell_Surface_Area / ConvexHull_Cell_Surface_Area\n"
                "Takes a value of 1 for convex shapes, and increases with the complexity of the volume.");
		  setIcon(QIcon(":/images/LobynessPlugin.png"));

    }

      bool run(){
        Mesh *m = currentMesh();
        return run(m);
      }

      bool run(Mesh *m);

   };

class Lobyness3DConvexVolume : public Process
  {
  public:
    Lobyness3DConvexVolume(const Process& process) : Process(process) {
      setName("Mesh/Heat Map/Measures 3D/Lobeyness/Solidity");
      setDesc("Lobyness Plugin 3D Version\n"
                "Solidity: Volume / ConvexHull_Volume");
      setIcon(QIcon(":/images/LobynessPlugin.png"));
    }

      bool run(){
        Mesh *m = currentMesh();
        return run(m);
      }

      bool run(Mesh *m);

  };

class Lobyness3DSphericity : public Process
  {
  public:
    Lobyness3DSphericity(const Process& process) : Process(process) {
      setName("Mesh/Heat Map/Measures 3D/Lobeyness/Sphericity");
      setDesc("Lobyness Plugin 3D Version\n"
                "Sphericity: (36 * PI * Volume^2) / (SurfaceArea^3).");
      setIcon(QIcon(":/images/LobynessPlugin.png"));

    }

      bool run(){
        Mesh *m = currentMesh();
        return run(m);
      }

      bool run(Mesh *m);
  };

class Lobyness3DCompactness : public Process
  {
  public:
    Lobyness3DCompactness(const Process& process) : Process(process) {
      setName("Mesh/Heat Map/Measures 3D/Lobeyness/Compactness");
      setDesc("Lobyness Plugin 3D Version\n"
                "Enclosing Sphere: ((4 * PI * (Longest diameter/2))");
      setIcon(QIcon(":/images/LobynessPlugin.png"));

    }

      bool run(){
        Mesh *m = currentMesh();
        return run(m);
      }

      bool run(Mesh *m);
  };


// class Lobyness3DLargestEmptySphere : public Process
//   {
//   public:
//     Lobyness3DLargestEmptySphere(const Process& process) : Process(process) {}

//       bool run(const QStringList &parms){
//         Mesh *m = currentMesh();
//         return run(m);
//       }

//     bool run(Mesh *m);
//     QString name() const { return "Mesh/Heat Map/Measures 3D/Lobyness/LargestEmptySphere"; }
//     QString description() const { return "Lobyness Plugin 3D Version\n"
//                 "LargestEmptySphere: diameter of the largest sphere the volume can contain";
//                  }
//     QStringList parmNames() const { return QStringList(); }
//     QStringList parmDefaults() const { return QStringList();}
//     QStringList parmDescs() const { return QStringList() ;}
//     QIcon icon() const { return QIcon(":/images/LobynessPlugin.png"); }
//     bool isMeasureProcess() const { return true; }

//   };
 

}

#endif
