//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef MISC_PROCESS_ANIMATE_HPP
#define MISC_PROCESS_ANIMATE_HPP

#include <Quaternion.hpp>
#include <Process.hpp>

namespace mgx
{
  typedef std::vector<Quaternion> QuaternionVec;
  ///\addtogroup Misc Processes
  ///@{

  /**
   * \class AddKeyFrame MiscProcessAnimate.hpp <MiscProcessAnimate.hpp>
   *
   * Add a key frame to the animation
   */
  class mgxBase_EXPORT AnimationAddKeyFrame : public Process 
  {
  public:
    AnimationAddKeyFrame(const Process& process) : Process(process) 
    {
      setName("Misc/Animate/Add Key Frame");
      setDesc("Add a key frame to the animation");
      setIcon(QIcon(":/images/AverageStacks.png"));

      addParm("Steps", "Steps from previous frame", "100");
    }
    bool run() { return run(parm("Steps").toInt()); }
    bool run(int steps);
  };

  /**
   * \class AnimationPlay MiscProcessAnimate.hpp <MiscProcessAnimate.hpp>
   *
   * Play the animation
   */
  class mgxBase_EXPORT AnimationPlay : public Process 
  {
  public:
    AnimationPlay(const Process& process) : Process(process) 
    {
      setName("Misc/Animate/Play");
      setDesc("Add a key frame to the animation");
      setIcon(QIcon(":/images/Process.png"));

      addParm("Loop", "Loop animation", "No", booleanChoice());
    }
    bool run() { return run(stringToBool(parm("Loop"))); }
    bool run(bool loop);
  };

  /**
   * \class AnimationClear MiscProcessAnimate.hpp <MiscProcessAnimate.hpp>
   *
   * Clear the animation
   */
  class mgxBase_EXPORT AnimationClear : public Process 
  {
  public:
    AnimationClear(const Process& process) : Process(process) 
    {
      setName("Misc/Animate/Clear");
      setDesc("Add a key frame to the animation");
      setIcon(QIcon(":/images/Clear.png"));
    }
    bool run();
  };

  /**
   * \class AnimationSave MiscProcessAnimate.hpp <MiscProcessAnimate.hpp>
   *
   * Save the animation
   */
  class mgxBase_EXPORT AnimationSave : public Process 
  {
  public:
    AnimationSave(const Process& process) : Process(process) 
    {
      setName("Misc/Animate/Save");
      setDesc("Save the key frames to a file");
      setIcon(QIcon(":/images/save.png"));

      addParm("File Name", "File name to save key frames", "Animation.txt");
    }
    bool run() { return run(parm("File Name")); }
    bool run(const QString &fileName);
  };
  
  /**
   * \class AnimationLoad MiscProcessAnimate.hpp <MiscProcessAnimate.hpp>
   *
   * Load the animation
   */
  class mgxBase_EXPORT AnimationLoad : public Process 
  {
  public:
    AnimationLoad(const Process& process) : Process(process) 
    {
      setName("Misc/Animate/Load");
      setDesc("Load the key frames to a file");
      setIcon(QIcon(":/images/open.png"));

      addParm("File Name", "File name to load key frames", "Animation.txt");
    }
    bool run() { return run(parm("File Name")); }
    bool run(const QString &fileName);
  };
///@}
}

#endif
