//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef PROGRESS_H
#define PROGRESS_H

#include <Config.hpp>

#include <QEvent>
#include <QList>
#include <QMutex>
#include <QObject>
#include <QPointer>
#include <QString>
#include <QProgressBar>
#include <QLabel>
#include <QToolBar>
#include <QPushButton>
#include <QtCore>
#include <string>

namespace mgx 
{
  // Progress class
  class mgx_EXPORT Progress : public QObject 
  {
    Q_OBJECT
  
    QWidget* _parent;
    QProgressBar* _progressBar;
    QLabel* _progressText;
    QPushButton* _progressStop;
    static Progress* _instance;
  
    bool _canceled;
    int _step;
    int _steps;
    QString _msg;
  
  public:
    Progress();
    ~Progress();

    bool processEvent(QEvent *e);
    void setupProgress(QWidget* parent, QToolBar* progressToolBar);

    void start(const QString& msg, int steps, bool allowCancel = true);
    void start(const char* msg, int steps, bool allowCancel = true)
    {
      start(QString::fromLocal8Bit(msg), steps, allowCancel);
    }
    void start(std::string msg, int steps, bool allowCancel = true)
    {
      start(QString::fromStdString(msg), steps, allowCancel);
    }
    bool advance(int step = -1);
    bool canceled();
    void cancel();
    void stop();
    void setMax(int steps);
    void setMsg(const QString& msg);
    void setMsg(const char* msg) { setMsg(QString::fromLocal8Bit(msg)); }
    void setMsg(std::string msg) { setMsg(QString::fromStdString(msg)); }
  
    static Progress& instance();

  private:
    QString makeMsg() const;
  };

  mgx_EXPORT void progressStart(const QString& msg, int steps = 0, bool allowCancel = true);
  mgx_EXPORT void progressStart(const char* msg, int steps = 0, bool allowCancel = true);
  mgx_EXPORT void progressStart(const std::string &msg, int steps = 0, bool allowCancel = true);
  mgx_EXPORT void progressStop();
  mgx_EXPORT bool progressAdvance(int step = -1);
  mgx_EXPORT bool progressCanceled();
  mgx_EXPORT void progressCancel();
  mgx_EXPORT void progressSetMsg(const QString &msg);
  mgx_EXPORT void progressSetMsg(const char *msg);
  mgx_EXPORT void progressSetMsg(const std::string &msg);
  mgx_EXPORT void progressSetMax(int steps = 0);
}
#endif
