//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef TIE_HPP
#define TIE_HPP
/**
 * \file Tie.hpp
 *
 * Defines the tie function
 */

#include <Config.hpp>

#include <utility>

namespace mgx 
{
  // Code taken from the Boost library !
  /**
   * \class refpair Tie.hpp <Tie.hpp>
   *
   * Class used to hold references for the tie() function.
   */
  template <typename T, typename U> struct refpair {
    typedef T first_type;
    typedef U second_type;
  
    /// Construct a pair of references to \c x and \c y.
    refpair(T& x, U& y) : first(x) , second(y) {}
    /// Construct a copy.
    refpair(refpair const& rp) : first(rp.first), second(rp.second) {}
  
    /// Assign the values of \c p to the references in this pair.
    refpair& operator=(std::pair<T, U> const& p)
    {
      first = p.first;
      second = p.second;
      return *this;
    }
  
    /// The first member of the pair.
    T& first;
    /// The second member of the pair.
    U& second;
  };
  
  /**
   * Tie two variables to the values of a pair.
   *
   * Example:
   * \code
   * std::pair<int,double> p(1,2.5);
   * int a;
   * double b;
   * tie(a,b) = p;
   * \endcode
   *
   * At the end, \c a is \c 1 and \c b is \c 2.5
   */
  template <typename T, typename U> inline refpair<T, U> tie(T& x, U& y) 
	{
    return refpair<T, U>(x, y);
  }
}
#endif
