//
// This file is part of MorphoGraphX - https://www.MorphoGraphX.org  (@RichardSmithLab)
//
// MorphoGraphX development is led by the Richard S. Smith lab at the John Innes Centre, Norwich, UK
//
// If you use MorphoGraphX in your work, please cite:
//   https://doi.org/10.7554/eLife.72601
//
// For support please see the image.sc forum:
//   https://forum.image.sc/tag/MorphoGraphX
//
// MorphoGraphX is copyright by its authors, contributors, and/or their employers.
//
// MorphoGraphX is free software, and is licensed under the terms of the 
// GNU General Public License https://www.gnu.org/licenses/.
//
#ifndef XPIWITProcess_HPP
#define XPIWITProcess_HPP

#include <XPIWITConfig.hpp>
#include <Process.hpp>
#include <Information.hpp>
#include <Progress.hpp>

namespace mgx
{
  ///\addtogroup Process
  ///@{
  /**
   * \class TWANGSegment XPIWITProcess.hpp <XPIWITProcess.hpp>
   *
   * Use XPIWIT with to do a TWANG nuclei segmentation 
   */
  class XPIWIT_EXPORT TwangSegment : public Process 
  {
  public:
    TwangSegment(const Process &process) : Process(process) 
    {
      setName("Stack/XPIWIT/TWANG Nuclei Segment");
      setDesc("Use XPIWIT to perform Nuclei segmentation with the TWANG pipeline.\n"
              "If you use this process in your work, please cite:\n"
              "  - Bartschat et al. 2016 XPIWIT - An XML Pipeline Wrapper for the Insight Toolkit, Bioinformatics 32(2).\n"
              "  - Stegmaier et al. 2014 Fast Segmentation of Stained Nuclei in Terabyte-Scale,\n"
              "    Time Resolved 3D Microscopy Image Stacks, PLoS One. 2014 9(2) e90036.");
      setIcon(QIcon(":/images/XPIWIT.png"));
      
      addParm("Min Sigma", "Minimum sigma for LoGScaleSpaceMaximumProjectionFilter", "4.0");
      addParm("Max Sigma", "Maximum sigma for LoGScaleSpaceMaximumProjectionFilter", "5.0");
      addParm("Std Dev Mult", "Multiplier for the weighting kernel standard deviation for ExtractLocalExtremaFilter", "1.0");
      addParm("Neighborhood Radius", "The neighborhood radius of the local extrema search for ExtractLocalExtremaFilter", "3");
	  }
  
    bool run()
    {
      Stack* stack = currentStack();
      Store* input = currentStack()->currentStore();
      Store* output = currentStack()->work();
      return run(stack, input, output);
    }
  
    bool run(Stack *stack, Store* input, Store* output);
  };

  /**
   * \class XPIWITRun XPIWITProcess.hpp <XPIWITProcess.hpp>
   *
   * Run XPIWIT 
   */
  class XPIWIT_EXPORT XPIWITRun : public Process 
  {
  public:
    XPIWITRun(const Process &process) : Process(process) 
    {
      setName("Stack/XPIWIT/XPIWIT Run");
      setDesc("Run XPIWIT in graphical mode to create your own ITK pipelines.\n"
              "If you use XPIWIT in your work, please cite:\n"
              "  Bartschat et al. 2016 XPIWIT - An XML Pipeline Wrapper for the Insight Toolkit, Bioinformatics 32(2).");
      setIcon(QIcon(":/images/XPIWIT.png"));
	  }
  
    bool run();
  };
  
  /**
   * \class XPIWITHelp XPIWITProcess.hpp <XPIWITProcess.hpp>
   *
   * Display XPIWIT help file 
   */
  class XPIWIT_EXPORT XPIWITHelp : public Process 
  {
  public:
    XPIWITHelp(const Process &process) : Process(process) 
    {
      setName("Stack/XPIWIT/XPIWIT Help");
      setDesc("View XPIWIT user guide.\n"
              "If you use XPIWIT in your work, please cite:\n"
              "  Bartschat et al. 2016 XPIWIT - An XML Pipeline Wrapper for the Insight Toolkit, Bioinformatics 32(2).");
      setIcon(QIcon(":/images/XPIWIT.png"));
	  }
  
    bool run();
  };

  ///@}
}
#endif
